/*!
 * @nuxt/cli v2.16.0 (c) 2016-2023
 * Released under the MIT License
 * Repository: https://github.com/nuxt/nuxt.js
 * Website: https://nuxtjs.org
*/
'use strict';

const consola = require('consola');
const stdEnv = require('std-env');
const chalk = require('chalk');
const index = require('./cli-index.js');
const prettyBytes = require('pretty-bytes');

function getMemoryUsage() {
  const { heapUsed, rss } = process.memoryUsage();
  return { heap: heapUsed, rss };
}
function getFormattedMemoryUsage() {
  const { heap, rss } = getMemoryUsage();
  return `Memory usage: ${chalk.bold(prettyBytes(heap))} (RSS: ${prettyBytes(rss)})`;
}
function showMemoryUsage() {
  consola.info(getFormattedMemoryUsage());
}

function showBanner(nuxt, showMemoryUsage = true) {
  if (stdEnv.isTest) {
    return;
  }
  if (stdEnv.isMinimal) {
    for (const listener of nuxt.server.listeners) {
      consola.info("Listening on: " + listener.url);
    }
    return;
  }
  const titleLines = [];
  const messageLines = [];
  const { bannerColor, badgeMessages } = nuxt.options.cli;
  titleLines.push(`${chalk[bannerColor].bold("Nuxt")} @ ${nuxt.constructor.version || "exotic"}
`);
  const label = (name) => chalk.bold.cyan(`\u25B8 ${name}:`);
  const isDev = nuxt.options.dev;
  let _env = isDev ? "development" : "production";
  if (process.env.NODE_ENV !== _env) {
    _env += ` (${chalk.cyan(process.env.NODE_ENV)})`;
  }
  titleLines.push(`${label("Environment")} ${_env}`);
  const isSSR = nuxt.options.render.ssr;
  const rendering = isSSR ? "server-side" : "client-side";
  titleLines.push(`${label("Rendering")}   ${rendering}`);
  const target = nuxt.options.target || "server";
  titleLines.push(`${label("Target")}      ${target}`);
  if (showMemoryUsage) {
    titleLines.push("\n" + getFormattedMemoryUsage());
  }
  for (const listener of nuxt.server.listeners) {
    messageLines.push(chalk.bold("Listening: ") + chalk.underline.blue(listener.url));
  }
  if (badgeMessages.length) {
    messageLines.push("", ...badgeMessages);
  }
  process.stdout.write(index.successBox(messageLines.join("\n"), titleLines.join("\n")));
}

exports.showBanner = showBanner;
exports.showMemoryUsage = showMemoryUsage;
