/*!
 * @nuxt/cli v2.16.0 (c) 2016-2023
 * Released under the MIT License
 * Repository: https://github.com/nuxt/nuxt.js
 * Website: https://nuxtjs.org
*/
'use strict';

const util = require('util');
const consola = require('consola');
const lodash = require('lodash');
const index = require('./cli-index.js');
require('@nuxt/utils');
require('@nuxt/config');
require('path');
require('exit');
require('chalk');
require('std-env');
require('wrap-ansi');
require('boxen');
require('minimist');
require('hable');
require('defu');
require('semver');
require('fs');
require('execa');

const webpack = {
  name: "webpack",
  description: "Inspect Nuxt webpack config",
  usage: "webpack [query...]",
  options: {
    ...index.common,
    name: {
      alias: "n",
      type: "string",
      default: "client",
      description: "Webpack bundle name: server, client, modern"
    },
    depth: {
      alias: "d",
      type: "string",
      default: 2,
      description: "Inspection depth"
    },
    colors: {
      type: "boolean",
      default: process.stdout.isTTY,
      description: "Output with ANSI colors"
    },
    dev: {
      type: "boolean",
      default: false,
      description: "Inspect development mode webpack config"
    }
  },
  async run(cmd) {
    const { name } = cmd.argv;
    const queries = [...cmd.argv._];
    const config = await cmd.getNuxtConfig({ dev: cmd.argv.dev, server: false });
    const nuxt = await cmd.getNuxt(config);
    const builder = await cmd.getBuilder(nuxt);
    const { bundleBuilder } = builder;
    const webpackConfig = bundleBuilder.getWebpackConfig(name);
    let queryError;
    const match = queries.reduce((result, query) => {
      const m = advancedGet(result, query);
      if (m === void 0) {
        queryError = query;
        return result;
      }
      return m;
    }, webpackConfig);
    const serialized = formatObj(match, {
      depth: parseInt(cmd.argv.depth),
      colors: cmd.argv.colors
    });
    consola.log(serialized + "\n");
    if (serialized.includes("[Object]")) {
      consola.info("You can use `--depth` or add more queries to inspect `[Object]` and `[Array]` fields.");
    }
    if (queryError) {
      consola.warn(`No match in webpack config for \`${queryError}\``);
    }
  }
};
function advancedGet(obj = {}, query = "") {
  let result = obj;
  if (!query || !result) {
    return result;
  }
  const [l, r] = query.split("=");
  if (!Array.isArray(result)) {
    return typeof result === "object" ? lodash.get(result, l) : result;
  }
  result = result.filter((i) => {
    const v = lodash.get(i, l);
    if (!v) {
      return false;
    }
    if (v === r || typeof v.test === "function" && v.test(r) || typeof v.match === "function" && v.match(r) || r && r.match(v)) {
      return true;
    }
    return false;
  });
  if (result.length === 1) {
    return result[0];
  }
  return result.length ? result : void 0;
}
function formatObj(obj, formatOptions) {
  if (!util.formatWithOptions) {
    return util.format(obj);
  }
  return util.formatWithOptions(formatOptions, obj);
}

exports.default = webpack;
